---@diagnostic disable: undefined-global

ServerBroadcaster = {}
ServerBroadcaster.MOD_NAME = g_currentModName or "FS25_EBC2"

ServerBroadcasterMessageEvent = {}
local ServerBroadcasterMessageEvent_mt = Class(ServerBroadcasterMessageEvent, Event)
InitEventClass(ServerBroadcasterMessageEvent, "ServerBroadcasterMessageEvent")

function ServerBroadcasterMessageEvent.emptyNew()
    local self = Event.new(ServerBroadcasterMessageEvent_mt)
    return self
end

function ServerBroadcasterMessageEvent.new(text)
    local self = ServerBroadcasterMessageEvent.emptyNew()
    self.text = text
    return self
end

function ServerBroadcasterMessageEvent:writeStream(streamId, connection)
    streamWriteString(streamId, self.text or "")
end

function ServerBroadcasterMessageEvent:readStream(streamId, connection)
    self.text = streamReadString(streamId)
    self:run(connection)
end

function ServerBroadcasterMessageEvent:run(connection)
    ServerBroadcaster:showMessage(self.text)
end

function ServerBroadcasterMessageEvent.sendToClients(text)
    if g_server ~= nil then
        g_server:broadcastEvent(ServerBroadcasterMessageEvent.new(text))
    end
end

local function trim(s)
    return (s:gsub("^%s+", ""):gsub("%s+$", ""))
end

function ServerBroadcaster:wrapBodyText(text, maxCharsPerLine, maxLines)
    local remaining = text or ""
    remaining = trim(remaining)
    if remaining == "" then
        return "", ""
    end

    local lines = {}

    while remaining ~= "" and #lines < maxLines do
        if #remaining <= maxCharsPerLine then
            table.insert(lines, trim(remaining))
            remaining = ""
        else
            local chunk = remaining:sub(1, maxCharsPerLine)
            local lastSpace = chunk:match("^.*() ")
            if lastSpace ~= nil and lastSpace > maxCharsPerLine * 0.4 then
                chunk = remaining:sub(1, lastSpace - 1)
                remaining = remaining:sub(lastSpace + 1)
            else
                remaining = remaining:sub(maxCharsPerLine + 1)
            end
            table.insert(lines, trim(chunk))
        end
    end

    if remaining ~= "" and #lines > 0 then
        lines[#lines] = trim(lines[#lines] .. " ...")
    end

    return lines[1] or "", lines[2] or ""
end

function ServerBroadcaster:loadMap(name)
    self.debug = false
    self.prefix = "Broadcast"
    self.intervalMs = 60000
    self.displayTimeMs = 5000
    self.messages = {}
    self.currentIndex = 1
    self.timer = 0

    self.autoReloadSeconds = 0
    self.reloadTimer = 0


    self.clientHeader = ""
    self.clientBody1 = ""
    self.clientBody2 = ""
    self.clientEndTime = 0
    self.bgOverlay = nil

    if g_server ~= nil then
        self:loadOrCreateSettings()

        addConsoleCommand("gsEBC2ReloadSettings", "Reload FS25_EBC2 Settings.xml", "consoleCommandReload", self)

        if self.debug then
            Logging.info("[%s] Broadcaster gestartet. %d Messages, Intervall %d ms",
                self.MOD_NAME, #self.messages, self.intervalMs)
        end
    end

    if g_client ~= nil then
        local modDir = g_modNameToDirectory[self.MOD_NAME] or ""
        local bgPath = Utils.getFilename("textures/info_bg.dds", modDir)
        if fileExists(bgPath) then
            self.bgOverlay = Overlay.new(bgPath, 0, 0, 1, 1)
        else
            print(("[%s] WARNUNG: textures/info_bg.dds nicht gefunden!"):format(self.MOD_NAME))
        end
    end
end

function ServerBroadcaster:deleteMap()
    if g_server ~= nil then
        removeConsoleCommand("gsEBC2ReloadSettings")
    end

    if self.bgOverlay ~= nil then
        self.bgOverlay:delete()
        self.bgOverlay = nil
    end
end

function ServerBroadcaster:update(dt)

    if g_server ~= nil then
        if self.messages ~= nil and #self.messages > 0 then
            self.timer = (self.timer or 0) + dt
            if self.timer >= self.intervalMs then
                self.timer = self.timer - self.intervalMs
                self:sendNextMessage()
            end
        end

        if self.autoReloadSeconds ~= nil and self.autoReloadSeconds > 0 then
            self.reloadTimer = (self.reloadTimer or 0) + dt
            if self.reloadTimer >= self.autoReloadSeconds * 1000 then
                self.reloadTimer = 0
                if self.debug then
                    print(("[%s] Auto-Reload Settings.xml"):format(self.MOD_NAME))
                end
                self:loadOrCreateSettings()
            end
        end
    end
end


function ServerBroadcaster:getSettingsPath()
    local basePath = getUserProfileAppPath()
    local settingsDir = basePath .. "modSettings/FS25_EBC2/"
    local settingsFile = settingsDir .. "Settings.xml"

    createFolder(basePath .. "modSettings/")
    createFolder(settingsDir)

    return settingsFile
end

function ServerBroadcaster:createDefaultSettings(settingsFile)
    local xml = createXMLFile("ServerBroadcasterSettings", settingsFile, "serverBroadcaster")

    setXMLBool(xml,  "serverBroadcaster.settings#debug", false)
    setXMLInt(xml,   "serverBroadcaster.settings#intervalSeconds", 60)
    setXMLString(xml,"serverBroadcaster.settings#prefix", "Broadcast")
    setXMLInt(xml,   "serverBroadcaster.settings#autoReloadSeconds", 0) -- 0 = aus

    setXMLString(xml, "serverBroadcaster.settings.messages.message(0)#text",
        "Willkommen auf unserem LS25 Server!")
    setXMLString(xml, "serverBroadcaster.settings.messages.message(1)#text",
        "Bitte haltet euch an die Serverregeln und habt Spaß beim Spielen!")
    setXMLString(xml, "serverBroadcaster.settings.messages.message(2)#text",
        "Tipp: Nutzt Verträge, um am Anfang schnell Geld zu verdienen.")

    saveXMLFile(xml)
    delete(xml)

    print(("[%s] Default Settings.xml erstellt: %s"):format(self.MOD_NAME, settingsFile))
end

function ServerBroadcaster:loadOrCreateSettings()
    local settingsFile = self:getSettingsPath()

    if not fileExists(settingsFile) then
        self:createDefaultSettings(settingsFile)
    end

    local xml = loadXMLFile("ServerBroadcasterSettings", settingsFile)
    if xml == nil then
        print(("[%s] Konnte Settings.xml nicht laden!"):format(self.MOD_NAME))
        return
    end

    self.debug = Utils.getNoNil(getXMLBool(xml, "serverBroadcaster.settings#debug"), false)

    local intervalSeconds = Utils.getNoNil(getXMLInt(xml, "serverBroadcaster.settings#intervalSeconds"), 60)
    if intervalSeconds < 5 then
        intervalSeconds = 5
    end
    self.intervalMs = intervalSeconds * 1000

    self.prefix = Utils.getNoNil(getXMLString(xml, "serverBroadcaster.settings#prefix"), "Broadcast")

    self.autoReloadSeconds = Utils.getNoNil(
        getXMLInt(xml, "serverBroadcaster.settings#autoReloadSeconds"),
        0
    )
    if self.autoReloadSeconds < 0 then
        self.autoReloadSeconds = 0
    end
    self.reloadTimer = 0

    self.messages = {}
    local i = 0
    while true do
        local key = string.format("serverBroadcaster.settings.messages.message(%d)", i)
        if not hasXMLProperty(xml, key) then
            break
        end
        local text = getXMLString(xml, key .. "#text")
        if text ~= nil and text ~= "" then
            table.insert(self.messages, text)
        end
        i = i + 1
    end

    delete(xml)

    if self.debug then
        print(("[%s] Settings geladen: debug=%s, interval=%d s, prefix='%s', autoReload=%d s, messages=%d"):format(
            self.MOD_NAME, tostring(self.debug), intervalSeconds, self.prefix, self.autoReloadSeconds, #self.messages))
    end
end

function ServerBroadcaster:sendNextMessage()
    if self.messages == nil or #self.messages == 0 then
        return
    end

    if self.currentIndex == nil or self.currentIndex < 1 then
        self.currentIndex = 1
    end
    if self.currentIndex > #self.messages then
        self.currentIndex = 1
    end

    local rawText = self.messages[self.currentIndex]
    self.currentIndex = self.currentIndex + 1

    local text
    if self.prefix ~= nil and self.prefix ~= "" then
        text = string.format("%s\n%s", self.prefix, rawText)
    else
        text = rawText
    end

    if self.debug then
        print(("[%s] Sende Message: %s"):format(self.MOD_NAME, text))
    end

    ServerBroadcasterMessageEvent.sendToClients(text)
    self:showMessage(text)
end


function ServerBroadcaster:showMessage(text)

    if g_client == nil
        and g_server ~= nil
        and g_currentMission ~= nil
        and g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    if text == nil or text == "" then
        return
    end

    local header, bodyFull = text:match("([^\n]+)\n(.+)")
    if header == nil then
        header = text
        bodyFull = ""
    end

    self.clientHeader = header
    self.clientBody1, self.clientBody2 = self:wrapBodyText(bodyFull or "", 90, 2)
    self.clientEndTime = (g_time or 0) + (self.displayTimeMs or 5000)
end

function ServerBroadcaster:draw()

    if g_client == nil
        and g_server ~= nil
        and g_currentMission ~= nil
        and g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    if (self.clientEndTime or 0) <= (g_time or 0) then
        return
    end

    if self.bgOverlay == nil then
        return
    end

    local uiScale = 1.0
    if g_gui ~= nil and g_gui.uiScale ~= nil then
        uiScale = g_gui.uiScale
    end

    local headerSize = 0.019 * uiScale
    local bodySize   = 0.014 * uiScale
    local paddingX   = 0.016 * uiScale
    local paddingY   = 0.012 * uiScale

    setTextBold(true)
    local headerWidth = getTextWidth(headerSize, self.clientHeader or "")
    setTextBold(false)
    local bodyWidth1 = getTextWidth(bodySize, self.clientBody1 or "")
    local bodyWidth2 = getTextWidth(bodySize, self.clientBody2 or "")

    local maxTextWidth = math.max(headerWidth, bodyWidth1, bodyWidth2, 0.1)
    local boxWidth = math.min(0.8, maxTextWidth + paddingX * 2)

    local numBodyLines = 0
    if self.clientBody1 ~= nil and self.clientBody1 ~= "" then numBodyLines = numBodyLines + 1 end
    if self.clientBody2 ~= nil and self.clientBody2 ~= "" then numBodyLines = numBodyLines + 1 end

    local headerHeight = headerSize * 1.4
    local bodyHeight   = bodySize   * 1.4 * numBodyLines
    local boxHeight    = paddingY * 2 + headerHeight + bodyHeight

    local centerX = 0.5

    local topY   = 0.955
    local bottomY = topY - boxHeight
    local posX   = centerX - boxWidth * 0.5

    self.bgOverlay:setDimension(boxWidth, boxHeight)
    self.bgOverlay:setPosition(posX, bottomY)
    self.bgOverlay:render()

    local curY

    setTextAlignment(RenderText.ALIGN_CENTER)
    setTextBold(true)

    curY = topY - paddingY - headerSize
    renderText(centerX, curY, headerSize, self.clientHeader or "")

    setTextBold(false)

    if self.clientBody1 ~= nil and self.clientBody1 ~= "" then
        curY = curY - headerSize * 1.4
        renderText(centerX, curY, bodySize, self.clientBody1)
    end

    if self.clientBody2 ~= nil and self.clientBody2 ~= "" then
        curY = curY - bodySize * 1.4
        renderText(centerX, curY, bodySize, self.clientBody2)
    end

    setTextAlignment(RenderText.ALIGN_LEFT)
end


function ServerBroadcaster:consoleCommandReload()
    if g_server == nil then
        return "FS25_EBC2: Reload nur auf dem Server/Host möglich."
    end

    self:loadOrCreateSettings()
    return "FS25_EBC2: Settings.xml wurde neu geladen."
end

addModEventListener(ServerBroadcaster)
