-- ============================================================================
-- AutoOpen StopBlocker Integration - Verhindert vorzeitiges Losfahren
-- 
-- Erweitert AutoOpenCore.lua um dynamische stopTrigger Funktionalität
-- Verhindert Kollisionen bei halb-geöffneten Toren (besonders Rolltore)
-- *** Version 1.1.0.0: EINFACHE MP-Synchronisation - Clients können Events senden ***
-- ============================================================================

-- Erweiterte AutoOpen Funktionen für StopBlocker
AutoOpen.stopBlocker = {}

-- StopBlocker Konfiguration
AutoOpen.stopBlocker.BLOCKER_FILE = "shared/stopBlocker.i3d"
AutoOpen.stopBlocker.CLEANUP_DELAY = 1000 -- 1 Sekunde nach Animation

-- ============================================================================
-- StopBlocker Management (EINFACHE MP-LÖSUNG)
-- ============================================================================

-- Lade StopBlocker an der Trigger Position
function AutoOpen.stopBlocker:loadStopBlocker(animatedObject, triggerId)
    -- Sichere Mod Directory Ermittlung mit Fallbacks
    local modDirectory = g_currentModDirectory
    
    -- Fallback 1: Über g_autoOpenModName
    if modDirectory == nil or modDirectory == "" then
        if g_autoOpenModName and g_modManager then
            local modDesc = g_modManager:getModByName(g_autoOpenModName)
            if modDesc then
                modDirectory = modDesc.modDir
                debugPrint("Mod Directory über g_modManager gefunden: " .. modDirectory)
            end
        end
    end
    
    -- Fallback 2: Debug Info vom Skript
    if modDirectory == nil or modDirectory == "" then
        local scriptDir = debug.getinfo(1, "S").source:sub(2)
        if scriptDir then
            modDirectory = scriptDir:match("(.*/)")
            if modDirectory then
                -- Gehe zwei Verzeichnisse hoch (von /src/core/ zum Hauptverzeichnis)
                modDirectory = modDirectory:gsub("/src/core/[^/]*$", ""):gsub("\\src\\core\\[^\\]*$", "")
                debugPrint("Mod Directory über debug.getinfo gefunden: " .. modDirectory)
            end
        end
    end
    
    if modDirectory == nil or modDirectory == "" then
        debugPrint("FEHLER: Mod Directory nicht verfügbar!")
        return false
    end
    
    -- Normalisiere Verzeichnispfad (stelle sicher dass er mit / endet)
    if not modDirectory:match("[/\\]$") then
        modDirectory = modDirectory .. "/"
    end
    
    -- Lade die I3D Datei MIT PHYSICS
    local blockerPath = modDirectory .. self.BLOCKER_FILE
    debugPrint("Lade StopBlocker von: " .. blockerPath)
    
    if not fileExists(blockerPath) then
        debugPrint("FEHLER: StopBlocker I3D nicht gefunden: " .. blockerPath)
        return false
    end
    
    local blockerRootNode = loadI3DFile(blockerPath, true, false, false)  -- Physics = TRUE!
    if blockerRootNode == 0 then
        debugPrint("FEHLER: Konnte StopBlocker I3D nicht laden!")
        return false
    end
    
    -- Finde den Blocker Node in der geladenen Datei
    local blockerNode = getChildAt(blockerRootNode, 0)
    if blockerNode == 0 then
        debugPrint("FEHLER: Kein Blocker in StopBlocker gefunden")
        delete(blockerRootNode)
        return false
    end
    
    -- Positioniere den Blocker an der Trigger Position
    local triggerParent = getParent(triggerId)
    if triggerParent == 0 then
        triggerParent = getRootNode()
    end
    
    -- Link den Blocker als Child der Trigger-Parent Node
    link(triggerParent, blockerRootNode)
    
    -- Kopiere Transform vom Original Trigger und versetze 10cm nach unten
    local x, y, z = getTranslation(triggerId)
    local rx, ry, rz = getRotation(triggerId)
    setTranslation(blockerRootNode, x, y - 0.1, z)  -- 10 cm nach unten
    setRotation(blockerRootNode, rx, ry, rz)
    
    debugPrint("StopBlocker 10cm nach unten versetzt - Y: " .. (y - 0.1))
    
    -- Speichere Referenzen
    animatedObject.stopBlockerRootNode = blockerRootNode
    animatedObject.stopBlockerNode = blockerNode
    animatedObject.stopBlockerActive = false
    
    -- Initial unsichtbar machen
    setVisibility(animatedObject.stopBlockerRootNode, false)
    
    debugPrint("StopBlocker erfolgreich geladen und positioniert mit Physics")
    return true
end

-- Aktiviere StopBlocker - EINFACHE MP-LÖSUNG
function AutoOpen.stopBlocker:activateStopBlocker(animatedObject, fromNetwork)
    if animatedObject.stopBlockerActive then
        debugPrint("StopBlocker bereits aktiv")
        return true
    end
    
    -- EINFACH: Sende Event falls nicht von Netzwerk (JEDER kann senden!)
    local isMP = g_server ~= nil or g_client ~= nil
    if isMP and not fromNetwork then
        debugPrint("MP: Sende StopBlocker ACTIVATE Event")
        if AutoOpenStopBlockerEvent then
            AutoOpenStopBlockerEvent.sendEvent(animatedObject, AutoOpenStopBlockerEvent.TYPE_ACTIVATE)
        else
            debugPrint("WARNING: AutoOpenStopBlockerEvent nicht verfügbar!")
        end
        -- WICHTIG: Führe trotzdem lokal aus - Event sorgt für Synchronisation
    end
    
    -- Lade StopBlocker falls nicht vorhanden
    if not animatedObject.stopBlockerNode then
        local success = self:loadStopBlocker(animatedObject, animatedObject.triggerNode)
        if not success then
            debugPrint("FEHLER: Konnte StopBlocker nicht laden")
            return false
        end
    end
    
    -- Sichtbar machen
    setVisibility(animatedObject.stopBlockerRootNode, true)
    animatedObject.stopBlockerActive = true
    
    debugPrint("StopBlocker AKTIVIERT - Einfache MP-Lösung")
    return true
end

-- Deaktiviere StopBlocker - EINFACHE MP-LÖSUNG
function AutoOpen.stopBlocker:deactivateStopBlocker(animatedObject, fromNetwork)
    if not animatedObject.stopBlockerActive then
        debugPrint("StopBlocker bereits inaktiv")
        return false
    end
    
    -- EINFACH: Sende Event falls nicht von Netzwerk (JEDER kann senden!)
    local isMP = g_server ~= nil or g_client ~= nil
    if isMP and not fromNetwork then
        debugPrint("MP: Sende StopBlocker DEACTIVATE Event")
        if AutoOpenStopBlockerEvent then
            AutoOpenStopBlockerEvent.sendEvent(animatedObject, AutoOpenStopBlockerEvent.TYPE_DEACTIVATE)
        else
            debugPrint("WARNING: AutoOpenStopBlockerEvent nicht verfügbar!")
        end
        -- WICHTIG: Führe trotzdem lokal aus - Event sorgt für Synchronisation
    end
    
    -- I3D komplett löschen
    if animatedObject.stopBlockerRootNode then
        delete(animatedObject.stopBlockerRootNode)
        debugPrint("StopBlocker I3D komplett gelöscht")
    end
    
    -- Referenzen löschen
    animatedObject.stopBlockerRootNode = nil
    animatedObject.stopBlockerNode = nil
    animatedObject.stopBlockerActive = false
    
    debugPrint("StopBlocker DEAKTIVIERT - Einfache MP-Lösung")
    return true
end

-- Entferne StopBlocker komplett
function AutoOpen.stopBlocker:removeStopBlocker(animatedObject)
    if not animatedObject.stopBlockerRootNode then
        return false
    end
    
    -- Deaktiviere erst falls noch aktiv (lokale Aktion, kein Event)
    if animatedObject.stopBlockerActive then
        -- Direkt löschen ohne Event
        if animatedObject.stopBlockerRootNode then
            delete(animatedObject.stopBlockerRootNode)
            debugPrint("StopBlocker I3D direkt gelöscht (Cleanup)")
        end
        
        animatedObject.stopBlockerRootNode = nil
        animatedObject.stopBlockerNode = nil
        animatedObject.stopBlockerActive = false
    end
    
    debugPrint("StopBlocker komplett entfernt")
    return true
end

-- ============================================================================
-- Gate State Management - UNVERÄNDERT
-- ============================================================================

-- Prüfe ob das Tor tatsächlich offen ist (nicht nur öffnend)
function AutoOpen.stopBlocker:isGateFullyOpen(animatedObject)
    if not animatedObject.animation then
        return false
    end
    
    -- Tor ist voll offen wenn Animation time = 1.0 und direction = 0 (oder > 0)
    local time = animatedObject.animation.time
    local direction = animatedObject.animation.direction
    
    local isFullyOpen = time >= 1.0 and direction >= 0
    debugPrint(string.format("Gate State Check: time=%.2f, direction=%d, isFullyOpen=%s", 
        time, direction, tostring(isFullyOpen)))
    
    return isFullyOpen
end

-- Prüfe ob das Tor tatsächlich geschlossen ist
function AutoOpen.stopBlocker:isGateFullyClosed(animatedObject)
    if not animatedObject.animation then
        return true -- Wenn keine Animation, dann ist es vermutlich geschlossen
    end
    
    -- Tor ist voll geschlossen wenn Animation time = 0.0 und direction = 0 (oder < 0)
    local time = animatedObject.animation.time
    local direction = animatedObject.animation.direction
    
    local isFullyClosed = time <= 0.0 and direction <= 0
    debugPrint(string.format("Gate State Check: time=%.2f, direction=%d, isFullyClosed=%s", 
        time, direction, tostring(isFullyClosed)))
    
    return isFullyClosed
end

-- Prüfe ob das Tor gerade öffnet (in Bewegung zum Öffnen)
function AutoOpen.stopBlocker:isGateOpening(animatedObject)
    if not animatedObject.animation then
        return false
    end
    
    local time = animatedObject.animation.time
    local direction = animatedObject.animation.direction
    
    -- Tor öffnet wenn direction > 0 und time < 1.0
    local isOpening = direction > 0 and time < 1.0
    debugPrint(string.format("Gate Opening Check: time=%.2f, direction=%d, isOpening=%s", 
        time, direction, tostring(isOpening)))
    
    return isOpening
end

-- ============================================================================
-- Integration in bestehende AutoOpen Funktionen - EINFACHE MP-LÖSUNG!
-- ============================================================================

-- Erweiterte autoOpenSet Funktion mit StopBlocker - EINFACH
local original_autoOpenSet = AutoOpen.autoOpenSet
function AutoOpen:autoOpenSet(animatedObject, open, noEventSend)
    debugPrint("autoOpenSet mit StopBlocker (Einfache MP-Lösung) - öffnen: " .. tostring(open))
    
    -- WICHTIGER FIX: Prüfe ob sich der State tatsächlich ändert!
    local wasOpen = AutoOpen.stopBlocker:isGateFullyOpen(animatedObject)
    local wasClosed = AutoOpen.stopBlocker:isGateFullyClosed(animatedObject)
    
    debugPrint(string.format("Gate State Before: wasOpen=%s, wasClosed=%s, requesting open=%s", 
        tostring(wasOpen), tostring(wasClosed), tostring(open)))
    
    if open then
        -- Tor soll geöffnet werden
        
        -- NUR StopBlocker aktivieren wenn das Tor NICHT bereits voll offen ist!
        if not wasOpen then
            -- Tor wird geöffnet - prüfe ob StopBlocker benötigt wird (nur bei Fahrzeugen)
            local needsBlocker = false
            if animatedObject.trackedObjects then
                for objectId, _ in pairs(animatedObject.trackedObjects) do
                    local object = g_currentMission.nodeToObject[objectId]
                    if object and object.getOwnerFarmId then
                        needsBlocker = true
                        break
                    end
                end
            end
            
            if needsBlocker then
                debugPrint("StopBlocker wird benötigt - Tor öffnet sich")
                
                -- Lade StopBlocker falls noch nicht vorhanden
                if not animatedObject.stopBlockerNode then
                    AutoOpen.stopBlocker:loadStopBlocker(animatedObject, animatedObject.triggerNode or animatedObject.triggerId)
                end
                
                if animatedObject.stopBlockerNode then
                    -- EINFACH: Aktiviere direkt (jeder kann das!)
                    AutoOpen.stopBlocker:activateStopBlocker(animatedObject, false) -- false = nicht von Netzwerk
                    
                    -- Timer für automatische Deaktivierung setzen
                    local animationTime = animatedObject.animation and animatedObject.animation.duration or 5000
                    animatedObject.stopBlockerDeactivateTimer = g_currentMission.time + animationTime + AutoOpen.stopBlocker.CLEANUP_DELAY
                    animatedObject:raiseActive()
                    debugPrint("StopBlocker Timer gesetzt")
                end
            else
                debugPrint("Kein StopBlocker benötigt - keine Fahrzeuge oder bereits offen")
            end
        else
            debugPrint("Tor ist bereits voll offen - KEIN StopBlocker aktiviert!")
        end
    else
        -- Tor wird geschlossen - StopBlocker SOFORT deaktivieren!
        -- Fahrzeuge müssen den Trigger verlassen können
        if animatedObject.stopBlockerNode then
            -- EINFACH: Deaktiviere direkt (jeder kann das!)
            AutoOpen.stopBlocker:deactivateStopBlocker(animatedObject, false) -- false = nicht von Netzwerk
            
            -- Timer löschen
            animatedObject.stopBlockerDeactivateTimer = nil
            debugPrint("StopBlocker deaktiviert - Fahrzeuge können den Trigger verlassen")
        end
    end
    
    -- Originale Funktion aufrufen
    original_autoOpenSet(self, animatedObject, open, noEventSend)
end

-- EINFACHE Update Funktion - JEDER kann deaktivieren
local original_update = AutoOpen.update
function AutoOpen:update(animatedObject, dt)
    -- EINFACH: Jeder Client prüft Timer und kann deaktivieren
    if animatedObject.stopBlockerDeactivateTimer then
        local currentTime = g_currentMission.time
        if currentTime > animatedObject.stopBlockerDeactivateTimer then
            debugPrint("StopBlocker Timer abgelaufen - prüfe Animation")
            
            -- Prüfe ob Animation wirklich beendet ist UND Tor voll offen ist
            local isFullyOpen = AutoOpen.stopBlocker:isGateFullyOpen(animatedObject)
            local isOpening = AutoOpen.stopBlocker:isGateOpening(animatedObject)
            
            debugPrint(string.format("Animation Check: isFullyOpen=%s, isOpening=%s", 
                tostring(isFullyOpen), tostring(isOpening)))
            
            if isFullyOpen and not isOpening then
                -- Tor ist voll offen und nicht mehr in Bewegung
                -- EINFACH: Deaktiviere direkt (Event wird automatisch gesendet)
                AutoOpen.stopBlocker:deactivateStopBlocker(animatedObject, false) -- false = nicht von Netzwerk
                animatedObject.stopBlockerDeactivateTimer = nil
                debugPrint("StopBlocker deaktiviert - Tor ist voll offen")
            else
                -- Animation läuft noch - Timer verlängern
                animatedObject.stopBlockerDeactivateTimer = currentTime + 1000
                debugPrint("Animation läuft noch - Timer verlängert")
            end
        end
    end
    
    -- Originale Update Funktion aufrufen
    original_update(self, animatedObject, dt)
end

-- Erweiterte triggerCallback mit StopBlocker Logic (UNVERÄNDERT)
local original_triggerCallback = AutoOpen.triggerCallback
function AutoOpen:triggerCallback(animatedObject, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
    -- Speichere Trigger Referenz für StopBlocker
    if not animatedObject.triggerNode then
        animatedObject.triggerNode = triggerId
    end
    
    -- Originale Trigger Callback
    original_triggerCallback(self, animatedObject, triggerId, otherId, onEnter, onLeave, onStay, anotherID)
end

-- ============================================================================
-- Cleanup Integration (UNVERÄNDERT)
-- ============================================================================

-- Erweiterte AnimatedObject delete für Cleanup
local originalAnimatedObjectDelete = AnimatedObject.delete
function AnimatedObject:delete()
    -- StopBlocker Cleanup
    if AutoOpen and AutoOpen.stopBlocker then
        AutoOpen.stopBlocker:removeStopBlocker(self)
    end
    
    -- Originale delete Funktion
    if originalAnimatedObjectDelete then
        originalAnimatedObjectDelete(self)
    end
end

debugPrint("AutoOpen StopBlocker System geladen - physisches Hindernis System mit EINFACHER MP-Synchronisation v1.1.0.2")