-- ============================================================================
-- AutoOpenStopBlockerEvent.lua - Multiplayer Event System for StopBlocker
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Multiplayer synchronization for StopBlocker operations
-- - Network event broadcasting for StopBlocker activation/deactivation
-- - Client/Server communication for collision objects
-- ============================================================================

AutoOpenStopBlockerEvent = {}
local AutoOpenStopBlockerEvent_mt = Class(AutoOpenStopBlockerEvent, Event)

InitEventClass(AutoOpenStopBlockerEvent, "AutoOpenStopBlockerEvent")

-- Event Types
AutoOpenStopBlockerEvent.TYPE_ACTIVATE = 1
AutoOpenStopBlockerEvent.TYPE_DEACTIVATE = 2

function AutoOpenStopBlockerEvent.emptyNew()
    local self = Event.new(AutoOpenStopBlockerEvent_mt)
    return self
end

function AutoOpenStopBlockerEvent.new(animatedObject, eventType)
    local self = AutoOpenStopBlockerEvent.emptyNew()
    self.animatedObject = animatedObject
    self.eventType = eventType
    return self
end

function AutoOpenStopBlockerEvent:readStream(streamId, connection)
    self.animatedObject = NetworkUtil.readNodeObject(streamId)
    self.eventType = streamReadUInt8(streamId)
    self:run(connection)
end

function AutoOpenStopBlockerEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.animatedObject)
    streamWriteUInt8(streamId, self.eventType)
end

function AutoOpenStopBlockerEvent:run(connection)
    if self.animatedObject ~= nil and AutoOpen and AutoOpen.stopBlocker then
        if self.eventType == AutoOpenStopBlockerEvent.TYPE_ACTIVATE then
            debugPrint("MP Event: StopBlocker ACTIVATE received")
            AutoOpen.stopBlocker:activateStopBlocker(self.animatedObject, true) -- true = from network
        elseif self.eventType == AutoOpenStopBlockerEvent.TYPE_DEACTIVATE then
            debugPrint("MP Event: StopBlocker DEACTIVATE received")
            AutoOpen.stopBlocker:deactivateStopBlocker(self.animatedObject, true) -- true = from network
        end
    end
end

-- Send StopBlocker event to all clients
function AutoOpenStopBlockerEvent.sendEvent(animatedObject, eventType)
    local isMP = g_server ~= nil or g_client ~= nil
    if not isMP then
        debugPrint("SP mode - no StopBlocker event needed")
        return
    end

    debugPrint("Sending StopBlocker event: " .. (eventType == 1 and "ACTIVATE" or "DEACTIVATE"))

    if g_server ~= nil then
        -- Server: send to all clients
        g_server:broadcastEvent(AutoOpenStopBlockerEvent.new(animatedObject, eventType), nil, nil, animatedObject)
        debugPrint("StopBlocker event broadcasted to all clients")
    else
        -- Client: send to server
        g_client:getServerConnection():sendEvent(AutoOpenStopBlockerEvent.new(animatedObject, eventType))
        debugPrint("StopBlocker event sent to server")
    end
end

debugPrint("AutoOpenStopBlockerEvent System loaded", "LOADING")