-- ============================================================================
-- AutoOpenManager.lua - Main Manager for AutoOpen System
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Settings management and GUI integration
-- - Translation system integration
-- - Compatibility layer for existing code
-- - Server/Client state management
-- - Debug Settings Control
-- ============================================================================

AutoOpenManager = {}

local autoOpenManager_mt = Class(AutoOpenManager)

function AutoOpenManager.new(mission, i18n, modName, modDirectory, customMt)
    local self = setmetatable({}, customMt or autoOpenManager_mt)

    self:mergeModTranslations(i18n)

    self.modName = modName
    self.modDirectory = modDirectory

    self.isServer = mission:getIsServer()
    self.isClient = mission:getIsClient()
    self.mission = mission

    debugPrint("AutoOpen Manager - Initialization")

    -- AutoOpenSettingsManager
    local title = i18n:getText("autoOpen_settings_title", self.customEnvironment)
    self.settings = AutoOpenSettingsManager.new(title, self, modName, modDirectory)

    -- Create settings
    self:initializeSettings(i18n)

    debugPrint("AutoOpenManager initialized")
    return self
end

-- Initialize all AutoOpen settings
function AutoOpenManager:initializeSettings(i18n)
    debugPrint("Initializing AutoOpen settings...")

    -- Master Switch Setting
    local title = i18n:getText("autoOpen_master_title", self.customEnvironment)
    local tooltip = i18n:getText("autoOpen_master_tooltip", self.customEnvironment)
    self.settings:addSetting("AUTOOPEN_ENABLED", AutoOpenSettingsManager.TYPE_BINARY, title, tooltip, true)

    -- Vehicles Setting  
    title = i18n:getText("autoOpen_vehicles_title", self.customEnvironment)
    tooltip = i18n:getText("autoOpen_vehicles_tooltip", self.customEnvironment)
    self.settings:addSetting("AUTOOPEN_VEHICLES_ENABLED", AutoOpenSettingsManager.TYPE_BINARY, title, tooltip, true)

    -- Players Setting
    title = i18n:getText("autoOpen_players_title", self.customEnvironment) 
    tooltip = i18n:getText("autoOpen_players_tooltip", self.customEnvironment)
    self.settings:addSetting("AUTOOPEN_PLAYERS_ENABLED", AutoOpenSettingsManager.TYPE_BINARY, title, tooltip, true)

    -- Delay Setting with options
    local delayOptions = {
        i18n:getText("autoOpen_delay_1s", self.customEnvironment),
        i18n:getText("autoOpen_delay_2s", self.customEnvironment),
        i18n:getText("autoOpen_delay_3s", self.customEnvironment),
        i18n:getText("autoOpen_delay_5s", self.customEnvironment),
        i18n:getText("autoOpen_delay_10s", self.customEnvironment)
    }

    title = i18n:getText("autoOpen_delay_title", self.customEnvironment)
    tooltip = i18n:getText("autoOpen_delay_tooltip", self.customEnvironment)
    self.settings:addSetting("AUTOOPEN_CLOSE_DELAY", AutoOpenSettingsManager.TYPE_MULTIBOX, title, tooltip, 3, delayOptions)

    -- *** NEUE DEBUG SETTING ***
    title = i18n:getText("autoOpen_debug_title", self.customEnvironment)
    tooltip = i18n:getText("autoOpen_debug_tooltip", self.customEnvironment)
    self.settings:addSetting("AUTOOPEN_DEBUG_ENABLED", AutoOpenSettingsManager.TYPE_BINARY, title, tooltip, false)

    debugPrint("AutoOpen settings created: " .. #self.settings.settings .. " settings")
end

-- Called on delete
function AutoOpenManager:delete()
    debugPrint("AutoOpen Manager - Delete")
    
    if self.mission ~= nil and self.mission.messageCenter ~= nil then
        self.mission.messageCenter:unsubscribeAll(self)
    end

    if self.settings ~= nil then
        self.settings:saveToXMLFile()
        debugPrint("Settings saved before delete")
    end

    debugPrint("AutoOpenManager deleted")
end

-- ============================================================================
-- Settings Interface - Compatibility with AutoOpenCore
-- ============================================================================

-- Returns setting value - Compatibility with old AutoOpenMod API
function AutoOpenManager:getSetting(settingName)
    if not self.settings then
        debugPrint("WARNING: Settings not initialized!")
        return self:getDefaultSetting(settingName)
    end

    -- Mapping of old names to new names
    local settingMapping = {
        autoOpenEnabled = "AUTOOPEN_ENABLED",
        autoOpenVehiclesEnabled = "AUTOOPEN_VEHICLES_ENABLED", 
        autoOpenPlayersEnabled = "AUTOOPEN_PLAYERS_ENABLED",
        closeDelaySeconds = "AUTOOPEN_CLOSE_DELAY",
        debugEnabled = "AUTOOPEN_DEBUG_ENABLED"  -- *** NEUE MAPPING ***
    }

    local mappedName = settingMapping[settingName] or settingName:upper()
    local value = self.settings:getSetting(mappedName)

    -- Debug output (only if debug is enabled)
    if settingName ~= "debugEnabled" then -- Avoid infinite recursion
        debugPrint("Mapping Debug - Original: " .. tostring(settingName) .. " -> Mapped: " .. tostring(mappedName))
        debugPrint("Raw Value: " .. tostring(value) .. " (type: " .. type(value) .. ")")
    end

    -- Special handling for delay (index to value)
    if settingName == "closeDelaySeconds" and value then
        local delayValues = {1, 2, 3, 5, 10}
        local result = delayValues[value] or 3
        if settingName ~= "debugEnabled" then
            debugPrint("Delay mapped to: " .. tostring(result))
        end
        return result
    end

    -- Explicit nil check for boolean false values
    local finalResult
    if value ~= nil then
        finalResult = value
    else
        finalResult = self:getDefaultSetting(settingName)
    end
    
    if settingName ~= "debugEnabled" then
        debugPrint("Final Result: " .. tostring(finalResult))
    end
    return finalResult
end

-- Sets setting value - Compatibility with old AutoOpenMod API
function AutoOpenManager:setSetting(settingName, value)
    if not self.settings then
        debugPrint("WARNING: Settings not initialized!")
        return false
    end

    -- Mapping of old names to new names
    local settingMapping = {
        autoOpenEnabled = "AUTOOPEN_ENABLED",
        autoOpenVehiclesEnabled = "AUTOOPEN_VEHICLES_ENABLED",
        autoOpenPlayersEnabled = "AUTOOPEN_PLAYERS_ENABLED", 
        closeDelaySeconds = "AUTOOPEN_CLOSE_DELAY",
        debugEnabled = "AUTOOPEN_DEBUG_ENABLED"  -- *** NEUE MAPPING ***
    }

    local mappedName = settingMapping[settingName] or settingName:upper()

    -- Special handling for delay (value to index)
    if settingName == "closeDelaySeconds" then
        local delayValues = {1, 2, 3, 5, 10}
        for i, delayValue in ipairs(delayValues) do
            if delayValue == value then
                value = i
                break
            end
        end
    end

    self.settings:setSetting(mappedName, value)
    debugPrint("Setting set: " .. settingName .. " = " .. tostring(value))
    return true
end

-- Get default setting values
function AutoOpenManager:getDefaultSetting(settingName)
    local defaults = {
        autoOpenEnabled = true,
        autoOpenVehiclesEnabled = true,
        autoOpenPlayersEnabled = true,
        closeDelaySeconds = 3,
        debugEnabled = false  -- *** NEUE DEFAULT-EINSTELLUNG ***
    }
    debugPrint("Using default for " .. settingName .. ": " .. tostring(defaults[settingName]))
    return defaults[settingName]
end

-- Print all current settings for debugging
function AutoOpenManager:printCurrentSettings()
    debugPrint("AutoOpen Manager Settings")
    if self.settings then
        for _, setting in ipairs(self.settings.settings) do
            debugPrint(setting.name .. " = " .. tostring(setting.value))
        end
    else
        debugPrint("Settings not initialized!")
    end
end

-- ============================================================================
-- Translation Management
-- ============================================================================

-- Merge local internationalization texts into global internationalization
function AutoOpenManager:mergeModTranslations(i18n)
    -- Thanks for blocking the getfenv Giants..
    local modEnvMeta = getmetatable(_G)
    local env = modEnvMeta.__index

    local global = env.g_i18n.texts
    for key, text in pairs(i18n.texts) do
        global[key] = text
    end
    
    debugPrint("Mod translations merged")
end

-- ============================================================================
-- Global Compatibility
-- ============================================================================

-- Create global alias for compatibility with existing code
AutoOpenMod = AutoOpenManager

debugPrint("AutoOpenManager System loaded (Debug-Settings verfügbar)")