-- ============================================================================
-- AutoOpenSettingsManager.lua - Settings GUI and Management System
-- 
-- Author: GulliDeckel
-- Version: 1.1.0.0
-- 
-- Features:
-- - Complete GUI settings integration
-- - XML settings persistence
-- - Binary and multibox setting types
-- - Dynamic GUI element creation
-- - Settings validation and callbacks
-- ============================================================================

AutoOpenSettingsManager = {}

local autoOpenSettingsManager_mt = Class(AutoOpenSettingsManager)

AutoOpenSettingsManager.TYPE_BINARY = 0
AutoOpenSettingsManager.TYPE_MULTIBOX = 1

AutoOpenSettingsManager.CLONE_REF = {
    [AutoOpenSettingsManager.TYPE_BINARY] = "checkActiveSuspensionCamera",
    [AutoOpenSettingsManager.TYPE_MULTIBOX] = "multiCameraSensitivity"
}

local settingsDirectory = g_currentModSettingsDirectory
if Platform.isConsole then
    settingsDirectory = getUserProfileAppPath()
end

-- Create new instance of AutoOpenSettingsManager
function AutoOpenSettingsManager.new(title, target, modName, modDirectory, customMt)
    local self = setmetatable({}, customMt or autoOpenSettingsManager_mt)

    self.title = title
    self.target = target

    self.modName = modName
    self.modDirectory = modDirectory

    self.settings = {}
    self.settingsByName = {}
    self.settingsCreated = false

    self.settingsSaveDirectory = settingsDirectory .. "autoOpenSettings.xml"

    if Platform.isConsole then
        registerProfileFile(self.settingsSaveDirectory)
    else
        createFolder(settingsDirectory)
    end

    debugPrint("AutoOpenSettingsManager created: " .. self.settingsSaveDirectory)

    return self
end

-- Load settings from xml file
function AutoOpenSettingsManager:loadFromXML()
    debugPrint("Loading AutoOpen settings from XML...")
    
    local xmlFile = XMLFile.loadIfExists("AutoOpenSettingsXMLFile", self.settingsSaveDirectory, AutoOpenSettingsManager.xmlSchema)

    if xmlFile == nil then
        debugPrint("No AutoOpen settings file found - using defaults")
        return
    end

    xmlFile:iterate("settings.setting", function(_, settingKey)
        local name = xmlFile:getValue(settingKey .. "#name")

        local existingSetting = self.settingsByName[name]
        if existingSetting ~= nil then
            local value
            if existingSetting.type == AutoOpenSettingsManager.TYPE_BINARY then
                value = xmlFile:getValue(settingKey .. "#boolean", false)
            elseif existingSetting.type == AutoOpenSettingsManager.TYPE_MULTIBOX then
                value = xmlFile:getValue(settingKey .. "#integer", 1)
            end

            if value ~= nil then
                self:setSetting(name, value)
                debugPrint("Setting loaded: " .. name .. " = " .. tostring(value))
            end
        end
    end)

    xmlFile:delete()
    debugPrint("AutoOpen settings successfully loaded from XML")
end

-- Save settings to xml file
function AutoOpenSettingsManager:saveToXMLFile()
    debugPrint("Saving AutoOpen settings to XML...")
    
    local xmlFile = XMLFile.create("AutoOpenSettingsXMLFile", self.settingsSaveDirectory, "settings", AutoOpenSettingsManager.xmlSchema)

    if xmlFile == nil then
        debugPrint("ERROR: Could not create AutoOpen settings XML file!")
        return
    end

    local baseKey = "settings.setting"
    local i = 0

    for _, setting in ipairs(self.settings) do
        local settingKey = ("%s(%d)"):format(baseKey, i)

        xmlFile:setValue(settingKey .. "#name", setting.name)

        if setting.type == AutoOpenSettingsManager.TYPE_BINARY then
            xmlFile:setValue(settingKey .. "#boolean", setting.value)
        elseif setting.type == AutoOpenSettingsManager.TYPE_MULTIBOX then
            xmlFile:setValue(settingKey .. "#integer", setting.value)
        end

        i = i + 1
    end

    xmlFile:save(false, false)
    xmlFile:delete()
    debugPrint("AutoOpen settings successfully saved to XML")
end

-- Sets value of given setting by name
function AutoOpenSettingsManager:setSetting(name, value)
    local setting = self.settingsByName[name]

    if setting == nil then
        debugPrint("WARNING: AutoOpenSettingsManager.setSetting: Invalid setting name: " .. tostring(name))
        return
    end

    setting.value = value

    local messageType = MessageType.SETTING_CHANGED[name]
    if messageType ~= nil then
        g_messageCenter:publish(messageType, value)
    end

    debugPrint("AutoOpen setting changed: " .. name .. " = " .. tostring(value))
end

-- Returns value of given setting by name
function AutoOpenSettingsManager:getSetting(name)
    local setting = self.settingsByName[name]

    if setting == nil then
        debugPrint("WARNING: AutoOpenSettingsManager.getSetting: Invalid setting name: " .. tostring(name))
        return nil
    end

    return setting.value
end

-- Add new setting to manager
function AutoOpenSettingsManager:addSetting(name, type, title, toolTip, initValue, options, callback, callbackTarget)
    if name == nil or name == "" then
        debugPrint("ERROR: Could not add AutoOpen setting without name!")
        return
    end

    if type == nil then
        debugPrint("ERROR: Could not add AutoOpen setting without type!")
        return
    end

    if type == AutoOpenSettingsManager.TYPE_BINARY then
        if callback == nil then
            callback = "onSettingChangedBinaryOption"
        end
        if initValue == nil then
            initValue = false
        end
    elseif type == AutoOpenSettingsManager.TYPE_MULTIBOX then
        if callback == nil then
            callback = "onSettingChangedMultibox"
        end
        if initValue == nil then
            initValue = 1
        end
    end

    name = name:upper()

    local setting = {
        name = name,
        type = type,
        title = title,
        toolTip = toolTip,
        value = initValue,
        options = options,
        callback = callback,
        callbackTarget = callbackTarget or self
    }

    table.addElement(self.settings, setting)
    self.settingsByName[name] = self.settings[#self.settings]

    -- Initialize MessageType if not available
    if MessageType.SETTING_CHANGED == nil then
        MessageType.SETTING_CHANGED = {}
    end
    MessageType.SETTING_CHANGED[name] = nextMessageTypeId()

    debugPrint("AutoOpen setting added: " .. name)
end

-- ============================================================================
-- GUI Creation & Management
-- ============================================================================

-- Create new Gui setting element by setting
function AutoOpenSettingsManager.createGuiElement(settingsFrame, setting, target)
    local cloneRef = AutoOpenSettingsManager.CLONE_REF[setting.type]

    if cloneRef == nil then
        debugPrint("ERROR: No CLONE_REF for AutoOpen setting type: " .. tostring(setting.type))
        return nil
    end

    cloneRef = settingsFrame[cloneRef]

    if cloneRef == nil then
        debugPrint("ERROR: No clone reference found for AutoOpen: " .. tostring(AutoOpenSettingsManager.CLONE_REF[setting.type]))
        return nil
    end

    local element = cloneRef.parent:clone()
    element.id = setting.name .. "Box"

    local settingElement = element.elements[1]
    local settingTitle = element.elements[2]
    local toolTip = settingElement.elements[1]

    settingTitle:setText(setting.title)
    toolTip:setText(setting.toolTip)
    settingElement.id = setting.name
    settingElement.target = setting.callbackTarget or target
    settingElement:setCallback("onClickCallback", setting.callback)
    settingElement:setDisabled(false)

    if setting.type == AutoOpenSettingsManager.TYPE_BINARY then
        settingElement:setIsChecked(setting.value)

        if setting.options == nil then
            setting.options = {
                g_i18n:getText("ui_off"),
                g_i18n:getText("ui_on"),
            }
        end
    elseif setting.type == AutoOpenSettingsManager.TYPE_MULTIBOX then
        settingElement:setState(setting.value, false)
    end

    if setting.options ~= nil then
        settingElement:setTexts(setting.options)
    end

    element:reloadFocusHandling(true)

    debugPrint("AutoOpen GUI element created for: " .. setting.name)
    return element
end

-- Injects AutoOpen settings into the InGameMenuSettingsFrame
function AutoOpenSettingsManager.initGui(settingsFrame, element, modEnvironment)
    local settingsManager = modEnvironment.settings
    local settingsElements = settingsFrame[settingsManager.title]

    if settingsElements == nil and not settingsManager.settingsCreated then
        debugPrint("AutoOpen Settings GUI - Initialization start")
        
        -- Copy header by name ref
        local headerRef
        for _, _element in ipairs(settingsFrame.generalSettingsLayout.elements) do
            if _element.name == 'sectionHeader' then
                headerRef = _element
                break
            end
        end

        if headerRef ~= nil then
            local headerElement = headerRef:clone()
            headerElement.id = settingsManager.title
            headerElement:setText(settingsManager.title)
            settingsFrame.generalSettingsLayout:addElement(headerElement)
            debugPrint("AutoOpen header created: " .. settingsManager.title)
        else
            debugPrint("WARNING: No header reference found for AutoOpen!")
        end

        -- Create setting elements
        settingsElements = {}

        for _, setting in ipairs(settingsManager.settings) do
            local createdElement = AutoOpenSettingsManager.createGuiElement(settingsFrame, setting, settingsManager)

            if createdElement ~= nil then
                settingsElements[setting.name] = createdElement
                settingsFrame.generalSettingsLayout:addElement(createdElement)
                debugPrint("AutoOpen GUI element added: " .. setting.name)
            else
                debugPrint("ERROR: Could not create AutoOpen GUI element for: " .. setting.name)
            end
        end

        settingsFrame.generalSettingsLayout:invalidateLayout()
        settingsFrame[settingsManager.title] = settingsElements

        settingsManager.settingsCreated = true
        debugPrint("AutoOpen Settings GUI - Initialization complete")
    end
end

-- Updates the AutoOpen settings once the InGameMenuSettingsFrame is opened
function AutoOpenSettingsManager.updateGui(settingsFrame, modEnvironment)
    local settingsManager = modEnvironment.settings
    local settingsElements = settingsFrame[settingsManager.title]

    if settingsManager ~= nil and settingsElements ~= nil then
        for _, setting in ipairs(settingsManager.settings) do
            local element = settingsElements[setting.name]

            if element ~= nil then
                if setting.type == AutoOpenSettingsManager.TYPE_BINARY and element.setIsChecked ~= nil then
                    element:setIsChecked(setting.value)
                elseif setting.type == AutoOpenSettingsManager.TYPE_MULTIBOX and element.setState ~= nil then
                    element:setState(setting.value)
                else
                    debugPrint("WARNING: GUI element has no matching method for type: " .. tostring(setting.type))
                end
            end
        end
        debugPrint("AutoOpen settings GUI updated")
    end
end

-- ============================================================================
-- Callback Functions
-- ============================================================================

-- Called on binary option change
function AutoOpenSettingsManager:onSettingChangedBinaryOption(state, element)
    self:setSetting(element.id, element:getIsChecked())
    debugPrint("AutoOpen binary setting changed: " .. element.id .. " = " .. tostring(element:getIsChecked()))
end

-- Called on multibox change
function AutoOpenSettingsManager:onSettingChangedMultibox(state, element)
    self:setSetting(element.id, state)
    debugPrint("AutoOpen multibox setting changed: " .. element.id .. " = " .. tostring(state))
end

-- ============================================================================
-- XML Schema Definition
-- ============================================================================

g_xmlManager:addCreateSchemaFunction(function()
    AutoOpenSettingsManager.xmlSchema = XMLSchema.new("autoOpenSettingsManager")
end)

g_xmlManager:addInitSchemaFunction(function()
    local schema = AutoOpenSettingsManager.xmlSchema
    local settingKey = "settings.setting(?)"

    schema:register(XMLValueType.STRING, settingKey .. "#name", "Name of setting", nil, true)
    schema:register(XMLValueType.BOOL, settingKey .. "#boolean", "Boolean value of setting")
    schema:register(XMLValueType.INT, settingKey .. "#integer", "Integer value of setting")
end)

-- debugPrint("AutoOpenSettingsManager System loaded")