--[[
    LoggingUtil

    This file contains functions to print debug info, warnings and errors.

    DISCLAIMER: For this script to work properly with the printDev functions, the game needs to get started with the
                parameter "-devWarnings"! Otherwise the printDev functions will not print anything!

        @author:    BayernGamers
        @date:      12.11.2024
        @version:   2.1

        History:    v1.0 @30.11.2023 - initial implementation in FS 22
                    -------------------------------------------------------------------------------------------
                    v2.0 @12.11.2024 - convert to FS25
                    -------------------------------------------------------------------------------------------
                    v2.1 @01.04.2025 - added xml DevWarning functions

        License:    Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]


LoggingUtil = {}
local LoggingUtil_mt = Class(LoggingUtil)

LoggingUtil.DEBUG_LEVELS = {
    LOW = 0,
    MEDIUM = 1,
    HIGH = 2
}

-- Create a new LoggingUtil object
-- @param enableDebug: If true, debug messages will be printed
-- @param debugLevel: The debug level of the messages
-- @param luaName: The name of the lua file
-- @return: The Logger
function LoggingUtil.new(enableDebug, debugLevel, luaName)
    local self = setmetatable({}, LoggingUtil_mt)

    self.enableDebug = enableDebug
    self.debugLevel = debugLevel
    self.luaName = luaName

    return self
end

-- Print debug info
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevInfo(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devInfo(prefix .. message)
    end
end

-- Print debug warning
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevWarning(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devWarning(prefix .. message)
    end
end

-- Print debug error
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevError(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devError(prefix .. message)
    end
end

-- Print info
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printInfo(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.info(prefix .. message)
end

-- Print warning
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printWarning(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.warning(prefix .. message)
end

-- Print error
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printError(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.error(prefix .. message)
end

-- Print XML info
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLInfo(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlInfo(xmlFile, prefix .. message)
end

-- Print XML warning
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLWarning(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlWarning(xmlFile, prefix .. message)
end

-- Print XML error
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLError(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlError(xmlFile, prefix .. message)
end

-- Print XML info with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLInfo(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevInfo(xmlFile, prefix .. message)
    end
end

-- Print XML warning with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLWarning(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevWarning(xmlFile, prefix .. message)
    end
end

-- Print XML error with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLError(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevError(xmlFile, prefix .. message)
    end
end
